define(["moment", "lib/components/base/modal"], function (Moment, Modal) {
  return class Events {
    constructor(widget) {
      this.widget = widget;
      this.lastWeeklyReport = [];
      this.lastCSV = "";
    }

    settings() {
      let _this = this;

      // handle pipeline change to update statuses
      $(document).off("change", "#" + _this.widget.config.prefix + "-filters-pipeline_id-id");
      $(document).on("change", "#" + _this.widget.config.prefix + "-filters-pipeline_id-id", function () {
        let pid = $(this).val();
        let statuses = _this.widget.info._statusesByPipeline[pid] || [];
        let $status = $("#" + _this.widget.config.prefix + "-filters-status_id-id");
        let html = _this.widget.render({ ref: "/tmpl/controls/select.twig" }, {
          items: statuses,
          class_name: _this.widget.config.prefix + "__filters-status_id ",
          id: _this.widget.config.prefix + "-filters-status_id-id",
          name: "params[filters][status_id]",
        });
        $status.replaceWith(html);
      });

      // calculate metrics
      $(document).off("click", "#" + _this.widget.config.prefix + "-actions-calculate-id");
      $(document).on("click", "#" + _this.widget.config.prefix + "-actions-calculate-id", function () {
        _this.calculateMetrics();
      });

      // export CSV
      $(document).off("click", "#" + _this.widget.config.prefix + "-actions-export_csv-id");
      $(document).on("click", "#" + _this.widget.config.prefix + "-actions-export_csv-id", function () {
        if (_this.lastCSV && _this.lastCSV.length > 0) {
          let blob = new Blob([_this.lastCSV], { type: "text/csv;charset=utf-8;" });
          let url = URL.createObjectURL(blob);
          let a = document.createElement("a");
          a.href = url;
          a.download = "weekly_whatsapp_report.csv";
          document.body.appendChild(a);
          a.click();
          document.body.removeChild(a);
          URL.revokeObjectURL(url);
        } else {
          new Modal().showError(_this.widget.i18n("settings.errors.nothing_to_export"), false);
        }
      });
    }

    async calculateMetrics() {
      const _this = this;
      try {
        const params = (("#" + _this.widget.config.prefix + "-settings__form").toString());
        const cfg = ($(params).serializeJSON() || {}).params || {};

        const dateRangeDays = parseInt(((cfg.filters || {}).date_range) || 30);
        const now = Moment();
        const from = Moment(now).subtract(dateRangeDays, "days").startOf("day").unix();
        const to = Moment(now).endOf("day").unix();

        // Fetch leads
        const [allLeads, bookingLeads] = await Promise.all([
          _this.widget.kommo.listLeads({ from, to }),
          _this.widget.kommo.listLeads({ from, to, pipeline_id: (cfg.filters || {}).pipeline_id, status_id: (cfg.filters || {}).status_id })
        ]);

        const totalLeads = allLeads.length;

        // Channels classification
        const mapping = cfg.mapping || {};
        const channelsConfig = {
          "FB Ads": (mapping.fb_ads || "").split(",").map(s => s.trim().toLowerCase()).filter(Boolean),
          "IG Ads": (mapping.ig_ads || "").split(",").map(s => s.trim().toLowerCase()).filter(Boolean),
          "Web": (mapping.web || "").split(",").map(s => s.trim().toLowerCase()).filter(Boolean),
          "Orgánico": (mapping.organic || "").split(",").map(s => s.trim().toLowerCase()).filter(Boolean),
          "Directo": (mapping.direct || "").split(",").map(s => s.trim().toLowerCase()).filter(Boolean)
        };

        function getLeadTags(lead) {
          const tags = (((lead || {})._embedded || {}).tags || []).map(t => (t.name || "").toLowerCase());
          return tags;
        }

        function classifyChannel(lead) {
          const tags = getLeadTags(lead);
          for (const [channel, keys] of Object.entries(channelsConfig)) {
            if (keys.length === 0) continue;
            for (const k of keys) {
              if (tags.some(t => t.indexOf(k) > -1)) return channel;
            }
          }
          return "Otros";
        }

        const leadsByChannelMap = {};
        allLeads.forEach((ld) => {
          const ch = classifyChannel(ld);
          leadsByChannelMap[ch] = (leadsByChannelMap[ch] || 0) + 1;
        });

        // Bookings
        const totalBookings = bookingLeads.length;
        const bookingRate = totalLeads > 0 ? Math.round((totalBookings / totalLeads) * 100) : 0;

        // Conversions by campaign
        const cfCampaignId = ((cfg.mapping || {}).campaign_field_id) || "";
        const byCampaign = {};
        const byCampaignTotalLeads = {};

        function getCFValue(lead, fieldId) { return _this.widget.kommo.getCFValue(lead, fieldId); }

        allLeads.forEach(ld => {
          const val = cfCampaignId ? getCFValue(ld, cfCampaignId) : "";
          if (val && String(val).length > 0) {
            byCampaignTotalLeads[val] = (byCampaignTotalLeads[val] || 0) + 1;
          }
        });
        bookingLeads.forEach(ld => {
          const val = cfCampaignId ? getCFValue(ld, cfCampaignId) : "";
          if (val && String(val).length > 0) {
            byCampaign[val] = (byCampaign[val] || 0) + 1;
          }
        });
        const campaignRows = Object.keys(byCampaignTotalLeads).map(name => {
          const leads = byCampaignTotalLeads[name] || 0;
          const bookings = byCampaign[name] || 0;
          const rate = leads > 0 ? Math.round((bookings / leads) * 100) : 0;
          return { name, leads, bookings, rate };
        }).sort((a,b) => b.bookings - a.bookings);

        // Weekly WhatsApp report (last 7 days)
        const waTag = (mapping.whatsapp_tag || "whatsapp").toLowerCase();
        const weekFrom = Moment(now).subtract(7, "days").startOf("day").unix();
        const weekTo = to;
        const weekBookings = await _this.widget.kommo.listLeads({ from: weekFrom, to: weekTo, pipeline_id: (cfg.filters || {}).pipeline_id, status_id: (cfg.filters || {}).status_id });
        const weeklyByCampaign = {};
        weekBookings.forEach(ld => {
          const tags = getLeadTags(ld);
          const hasWa = tags.some(t => t.indexOf(waTag) > -1);
          if (!hasWa) return;
          const c = cfCampaignId ? getCFValue(ld, cfCampaignId) : "";
          if (c && String(c).length > 0) {
            weeklyByCampaign[c] = (weeklyByCampaign[c] || 0) + 1;
          }
        });
        const weeklyRows = Object.keys(weeklyByCampaign).map(k => ({ name: k, bookings: weeklyByCampaign[k] })).sort((a,b) => b.bookings - a.bookings);

        // Prepare CSV for weekly report
        let csv = "Campaign,Bookings (WhatsApp),From,To\n";
        weeklyRows.forEach(r => { csv += `"${r.name}",${r.bookings},${Moment.unix(weekFrom).format("YYYY-MM-DD")},${Moment.unix(weekTo).format("YYYY-MM-DD")}\n`; });
        _this.lastWeeklyReport = weeklyRows;
        _this.lastCSV = csv;

        // Render dashboard
        const leadsByChannel = Object.keys(leadsByChannelMap).map(name => ({ name, count: leadsByChannelMap[name] })).sort((a,b) => b.count - a.count);
        const html = _this.widget.templates.render("settings.dashboard", {
          prefix: _this.widget.config.prefix,
          langs: _this.widget.i18n("settings"),
          metrics: {
            total_leads: totalLeads,
            leads_by_channel: leadsByChannel,
            bookings: { total: totalBookings, rate: bookingRate },
            campaigns: campaignRows,
            weekly: { whatsapp_pauta: weeklyRows }
          }
        });

        $("#" + _this.widget.config.prefix + "-dashboard").html(html);
      } catch (ex) {
        _this.widget.debug.error(ex);
        new Modal().showError("Error calculating metrics", false);
      }
    }
  };
});
